from pydantic import BaseModel
import openpyxl.worksheet
import openpyxl
import os

from typing import Union

from excel.blocks import *


class ExcelEditor:

    def __init__(self):
        self.wb = None
        self.sheet = None
        self._cur_row = 1

    def create(self):
        """
        Create new workbook.
        """
        self.wb = openpyxl.Workbook()
        self.sheet = self.wb.active
        return self

    def open(self, filename: str, sheet_number: int = None) -> None:
        """
        Open Excel file.

        :param filename: xlsx filename
        :param sheet_number: sheet number
        """
        if filename and os.path.exists(filename) and os.path.isfile(filename):
            self.wb = openpyxl.load_workbook(filename=filename)
            if sheet_number:
                self.wb.active = sheet_number
            self.sheet = self.wb.active
        else:
            raise OSError(f"Not exist file with filename \"{filename}\"")

    def save(self, filename: str) -> None:
        self.wb.save(filename=filename)

    def _add_block(self, block, fmt: BaseModel) -> None:
        self._cur_row = block(self.sheet, self._cur_row).put(**fmt.dict())

    def add_blocks(self, blocks: Union[list, tuple], formats: Union[list, tuple]) -> None:
        """
        Add blocks on Excel sheet.

        :param blocks: block classes
        :param formats: block formats
        """
        assert len(blocks) == len(formats), ValueError("Each block need format")
        for block, fmt in zip(blocks, formats):
            self._add_block(block, fmt)
