import sys
from typing import Optional, List, Dict

from tqdm import tqdm

import openpyxl.worksheet.cell_range
from pydantic import BaseModel
from openpyxl.utils import get_column_letter
from openpyxl.styles import Font
from fontTools.ttLib import TTFont
import os

from settings import TableConf
from settings import COL_AUTOSIZE, COL_MIN_WIDTH, COL_MAX_WIDTH, FONTS_PATH
from tools.calibrate_normalization_coef import FontNormCoef
from tools.utils import timing
from .base import BaseBlock

# logger
from settings import LOGGER

LEFT_ROWS_PERCENT = 11
MERGED_LEFT_ROWS_PERCENT = 12
DATA_ROWS_PERCENT_MIN = 13
DATA_ROWS_PERCENT_MAX = 72
NOT_MERGED_CELLS_PERCENT_MIN = 73
NOT_MERGED_CELLS_PERCENT_MAX = 96
MERGED_CELLS_PERCENT = 100


class CellAutoResizeFormat(BaseModel):
    fonts_path: str = FONTS_PATH
    autosize: bool = COL_AUTOSIZE
    min_width: Optional[float] = COL_MIN_WIDTH
    max_width: Optional[float] = COL_MAX_WIDTH
    row_end_num: Optional[int] = None


class CellAutoResizeBlock(BaseBlock):
    """
    Auto resize block for cells in table range.
    Calculate length of each cell value and resize columns and rows.

    Algorithm for resizing: https://stackoverflow.com/questions/4190667/how-to-get-width-of-a-truetype-font-character-in-1200ths-of-an-inch-with-python
    """

    NORMALAZING_COEF: float = FontNormCoef.get_norm_coef()
    FONT_NAMES_WITH_PATH: dict = None
    ADDITIONAL_WIDTH: float = 1.  # additional width for perfect work

    def __init__(self, sheet, row_start):
        super().__init__(sheet, row_start)

        self.font_names_with_path = None
        self._merged_cells_set = set()

        # optimization extra variables
        self._text_hash_len = {}
        self._max_possible_symbol_len = None

    @classmethod
    def _update_font_paths(cls, fonts_path: str = None, **kwargs):
        """
        Update font paths.

        :param fonts_path: optional path to folder with different fonts, example: 'home/user/fonts'
        """
        if fonts_path:
            cls.FONT_NAMES_WITH_PATH = {os.path.splitext(file)[0].lower(): os.path.join(fonts_path, file)
                                        for file in os.listdir(fonts_path)}

    @classmethod
    def _get_fonttools_params_by_font_name(cls, font_name: str):
        """
        Get fontTools params for calculation symbol width.

        :param font_name: current text font name
        :return: two fontTools params
        """
        if not cls.FONT_NAMES_WITH_PATH:
            raise ValueError("Empty FONT_NAMES_WITH_PATH. Check font paths.")
        font_path = cls.FONT_NAMES_WITH_PATH.get(font_name.lower(), None)
        if font_path is None:
            raise ValueError(
                f"Can't find font with name {font_name.lower()}; available fonts: {cls.FONT_NAMES_WITH_PATH}")
        font = TTFont(font_path)
        t = font['cmap'].getcmap(3, 1).cmap
        s = font.getGlyphSet()
        return t, s

    @staticmethod
    def _add_bold_postfix(font_name: str) -> str:
        """
        Add bold postfix to font name.
        """
        return font_name + "_bold"

    def _calculate_len_text(self, total_len, font_size: float):
        return (total_len / self.NORMALAZING_COEF + self.ADDITIONAL_WIDTH) * font_size / FontNormCoef.DEFAULT_FONT_SIZE

    def _get_text_width(self, text: str, font_name: str, font_size: float, font_bold: bool = False,
                        use_hash: bool = True) -> float:
        """
        Get text width in px.

        :param text: string text
        :param font_name: font name of text
        :param font_size: font size of text
        :param font_bold: font bold of text
        :return: float length in px
        """
        if use_hash and text in self._text_hash_len:
            return self._text_hash_len[text]

        font_name = self._add_bold_postfix(font_name) if font_bold else font_name  # check is need bold postfix
        t, s = self._get_fonttools_params_by_font_name(font_name)

        if not self._max_possible_symbol_len:
            self._max_possible_symbol_len = 0
            for key in dict(s).keys():
                self._max_possible_symbol_len = max(0, s[key].width)

        total = 0
        for c in text:
            if ord(c) in t and t[ord(c)] in s:
                total += s[t[ord(c)]].width
            else:
                total += s['.notdef'].width

        len_ = self._calculate_len_text(total, font_size)

        if use_hash:
            self._text_hash_len[text] = len_

        return len_

    def _check_cell_is_merged(self, row_num: int, col_num: int) -> bool:
        """
        Check if cell is merged.

        :param row_num: row num
        :param col_num: col num
        :return: bool value
        """
        if not self._merged_cells_set:
            for mergedCell in self.sheet.merged_cells.ranges:
                mergedCell: openpyxl.worksheet.cell_range.CellRange
                self._merged_cells_set.update({pare for pare in mergedCell.cells})

        return True if (row_num, col_num) in self._merged_cells_set else False

    @timing
    def _update_not_merged_col_width(self, row_end_num: Optional[int],
                                     min_width: Optional[float], max_width: Optional[float]):
        """
        Update cols width. Iterate on each cell in each column and calculate max width if cell is not merged.
        Set new width for column with ranges *min_width* or *max_width* if set.

        :param row_end_num: optional last row number for table range
        :param min_width: optional minimum width
        :param max_width: optional maximum width
        """
        # LOGGER.info(f"Total columns: {self.sheet.max_column}")
        total_steps = row_end_num - 1
        total_columns = self.sheet.max_column
        for i_col, col in enumerate(self.sheet.columns):
            col_num = i_col + 1
            col_max_width = None
            total_percent = NOT_MERGED_CELLS_PERCENT_MIN + \
                            ((i_col / total_columns) *
                             (NOT_MERGED_CELLS_PERCENT_MAX - NOT_MERGED_CELLS_PERCENT_MIN))
            for i_row in tqdm(range(total_steps),
                              desc=f"Update not merged cells width, column {i_col}",
                              postfix=f"total progress: {total_percent:.0f}%",
                              file=sys.stdout):
                cell = col[i_row]

                if self._check_cell_is_merged(i_row + 1, col_num) or not cell.value:
                    continue

                value = str(cell.value)

                # optimization
                if col_max_width and self._max_possible_symbol_len:
                    text_width = self._calculate_len_text(
                        len(value) * self._max_possible_symbol_len,
                        cell.font.size)
                    if text_width > col_max_width:
                        text_width = self._get_text_width(value, cell.font.name, cell.font.size, cell.font.bold)
                    col_max_width = max(text_width, col_max_width)
                # simple calculation
                else:
                    text_width = self._get_text_width(value, cell.font.name, cell.font.size, cell.font.bold)
                    col_max_width = max(text_width, col_max_width) if col_max_width else text_width

            if not col_max_width:
                continue

            if max_width and col_max_width > max_width:
                col_max_width = max_width

            elif min_width and col_max_width < min_width:
                col_max_width = min_width

            self.sheet.column_dimensions[get_column_letter(col_num)].width = col_max_width

    @timing
    def _update_merged_col_width(self, max_width: Optional[float]):
        """
        Update cols width. Iterate on each cell in range between min_col and max_col for all merged regions,
        calculate total width for region, calculate text width, calculate diff width
        and add diff in each column in range.

        :param max_width: optional maximum width
        """
        for mergedCell in tqdm(self.sheet.merged_cells.ranges,
                               desc="Update merged cells width",
                               postfix=f"total progress: {MERGED_CELLS_PERCENT}%",
                               file=sys.stdout):
            mergedCell: openpyxl.worksheet.cell_range.CellRange
            cell = self.sheet.cell(row=mergedCell.min_row, column=mergedCell.min_col)

            value = cell.value
            if not value:
                continue

            total_width = sum(self.sheet.column_dimensions[get_column_letter(col_num)].width
                              for col_num in range(mergedCell.min_col, mergedCell.max_col))
            text_width = self._get_text_width(str(value), cell.font.name, cell.font.size, cell.font.bold)

            if total_width >= text_width:
                continue

            diff = text_width - total_width

            for col_num in range(mergedCell.min_col, mergedCell.max_col):
                tmp_col = self.sheet.column_dimensions[get_column_letter(mergedCell.min_col)]
                if max_width:
                    # add max extra width for each column until diff is used up
                    max_diff = max_width - tmp_col.width
                    tmp_col.width += min(max_diff, diff)
                    diff -= max_diff
                    if diff <= 0:
                        break
                else:
                    # add extra width for each column
                    tmp_col.width += diff * (mergedCell.max_col - mergedCell.min_col)

    def _update_col_width(self, row_end_num: Optional[int], min_width: Optional[float], max_width: Optional[float]):
        """
        Update cols width. First iterate on each cell in each column and update width,
        then iterate on each merged region.

        :param row_end_num: optional last row number for table range
        :param min_width: optional minimum width
        :param max_width: optional maximum width
        """

        # first update width for not merged cells
        self._update_not_merged_col_width(row_end_num, min_width, max_width)

        # check width for merged cells
        self._update_merged_col_width(max_width)

    @timing
    def put(self, autosize: bool, row_end_num: Optional[int] = None, fonts_path: Optional[str] = None,
            min_width: Optional[float] = None, max_width: Optional[float] = None, **kwargs):
        """
        Update columns' width and rows' height.

        :param autosize: bool change size
        :param row_end_num: optional last row number for table range
        :param fonts_path: optional path to folder with different fonts, example: 'home/user/fonts'
        :param min_width: optional minimum width
        :param max_width: optional maximum width
        :param kwargs: other params from pydantic format
        """

        self._update_font_paths(fonts_path, **kwargs)

        if autosize:
            self._update_col_width(row_end_num, min_width, max_width)


class TableFormat(BaseModel):
    top_dims: List
    left_dims: List
    data_header: List[List[Dict]]
    left_data: List[List[Dict]]
    data: List[List]

    header_font_size: Optional[int] = TableConf.HEADER_FONT_SIZE
    header_font_bold: Optional[bool] = TableConf.HEADER_FONT_BOLD
    header_align_h: Optional[str] = TableConf.HEADER_ALIGN_H
    header_align_v: Optional[str] = TableConf.HEADER_ALIGN_V
    header_wrap_text: Optional[bool] = TableConf.HEADER_WRAP_TEXT

    left_dim_align_h: Optional[str] = TableConf.LEFT_DIM_ALIGN_H
    left_dim_align_v: Optional[str] = TableConf.LEFT_DIM_ALIGN_V
    left_dim_wrap_text: Optional[bool] = TableConf.LEFT_WRAP_TEXT

    data_dim_align_h: Optional[str] = TableConf.DATA_DIM_ALIGN_H
    data_dim_align_v: Optional[str] = TableConf.DATA_DIM_ALIGN_V
    data_wrap_text: Optional[bool] = TableConf.DATA_WRAP_TEXT
    data_total_col_font_bold: Optional[bool] = TableConf.DATA_TOTAL_COL_FONT_BOLD
    data_total_row_font_bold: Optional[bool] = TableConf.DATA_TOTAL_ROW_FONT_BOLD
    data_number_format: Optional[str] = TableConf.DATA_NUMBER_FORMAT

    border_style: Optional[str] = TableConf.BORDER_STYLE
    border_color: Optional[str] = TableConf.BORDER_COLOR

    # table view, converts to dict!
    view_format: Optional[CellAutoResizeFormat] = None


class TableBlock(BaseBlock):
    HEADER_FONT_SIZE = None
    HEADER_FONT_BOLD = None
    HEADER_ALIGN_H = None
    HEADER_ALIGN_V = None
    HEADER_WRAP_TEXT = None

    LEFT_DIM_ALIGN_H = None
    LEFT_DIM_ALIGN_V = None
    LEFT_WRAP_TEXT = None

    DATA_DIM_ALIGN_H = None
    DATA_DIM_ALIGN_V = None
    DATA_WRAP_TEXT = None
    DATA_TOTAL_COL_FONT_BOLD = None
    DATA_TOTAL_ROW_FONT_BOLD = None

    BORDER_STYLE = None
    BORDER_COLOR = None

    def __init__(self, sheet, row_start):
        super().__init__(sheet, row_start)

        self.total_col_sheet_indexes = set()  # номера колонок с итогами в excel-странице
        self.total_row_sheet_indexes = set()  # номера строк с итогами в excel-странице

    @classmethod
    def update_font_settings(cls,
                             header_font_size: Optional[int] = None,
                             header_font_bold: Optional[bool] = None,
                             header_align_h: Optional[str] = None,
                             header_align_v: Optional[str] = None,
                             left_dim_align_h: Optional[str] = None,
                             left_dim_align_v: Optional[str] = None,
                             data_dim_align_h: Optional[str] = None,
                             data_dim_align_v: Optional[str] = None,
                             header_wrap_text: Optional[bool] = None,
                             left_dim_wrap_text: Optional[bool] = None,
                             data_wrap_text: Optional[bool] = None,
                             border_style: Optional[str] = None,
                             border_color: Optional[str] = None,
                             data_total_col_font_bold: Optional[bool] = None,
                             data_total_row_font_bold: Optional[bool] = None,
                             data_number_format: Optional[str] = None,
                             **kwargs):
        cls.HEADER_FONT_SIZE = header_font_size
        cls.HEADER_FONT_BOLD = header_font_bold
        cls.HEADER_ALIGN_H = header_align_h
        cls.HEADER_ALIGN_V = header_align_v
        cls.HEADER_WRAP_TEXT = header_wrap_text

        cls.LEFT_DIM_ALIGN_H = left_dim_align_h
        cls.LEFT_DIM_ALIGN_V = left_dim_align_v
        cls.LEFT_WRAP_TEXT = left_dim_wrap_text

        cls.DATA_DIM_ALIGN_H = data_dim_align_h
        cls.DATA_DIM_ALIGN_V = data_dim_align_v
        cls.DATA_WRAP_TEXT = data_wrap_text
        cls.DATA_TOTAL_COL_FONT_BOLD = data_total_col_font_bold
        cls.DATA_TOTAL_ROW_FONT_BOLD = data_total_row_font_bold
        cls.DATA_NUMBER_FORMAT = data_number_format

        cls.BORDER_STYLE = border_style
        cls.BORDER_COLOR = border_color

    @timing
    def _set_headers(self, top_dims: List, left_dims: List, data_header: List[List[Dict]]):
        row_curr = self.row_curr

        header_font = self._create_font(font_size=self.HEADER_FONT_SIZE,
                                        font_bold=self.HEADER_FONT_BOLD)
        header_align = self._create_alignment(align_h=self.HEADER_ALIGN_H,
                                              align_v=self.HEADER_ALIGN_V,
                                              wrap_text=self.HEADER_WRAP_TEXT)
        top_dims_position = ('top', 'left', 'right', 'bottom')

        # set top dims
        if not top_dims:
            cell = self._set_value(row=row_curr,
                                   col=1,
                                   value="",
                                   font=header_font,
                                   alignment=header_align)
            cell.border = self._create_border(positions=top_dims_position,
                                              style=self.BORDER_STYLE, color=self.BORDER_COLOR)
            self.sheet.merge_cells(start_row=row_curr, start_column=1,
                                   end_row=row_curr, end_column=len(left_dims))
            row_curr += 1
        else:
            for top_dim_value in top_dims:
                cell = self._set_value(row=row_curr,
                                       col=1,
                                       value=top_dim_value,
                                       font=header_font,
                                       alignment=header_align)
                cell.border = self._create_border(positions=top_dims_position,
                                                  style=self.BORDER_STYLE, color=self.BORDER_COLOR)
                self.sheet.merge_cells(start_row=row_curr, start_column=1,
                                       end_row=row_curr, end_column=len(left_dims) if left_dims else 1)
                row_curr += 1

        # set left dims
        left_dims_position = top_dims_position

        if not left_dims:
            cell = self._set_value(row=row_curr,
                                   col=1,
                                   value="",
                                   font=header_font,
                                   alignment=header_align)
            cell.border = self._create_border(positions=left_dims_position,
                                              style=self.BORDER_STYLE, color=self.BORDER_COLOR)
        else:
            for i, left_dim_value in enumerate(left_dims):
                cell = self._set_value(row=row_curr,
                                       col=1 + i,
                                       value=left_dim_value,
                                       font=header_font,
                                       alignment=header_align)
                cell.border = self._create_border(positions=left_dims_position,
                                                  style=self.BORDER_STYLE, color=self.BORDER_COLOR)

        # set data header
        row_curr = self.row_curr
        data_header_position = left_dims_position

        for data_header_row in data_header:

            for i, data_header_cell in enumerate(data_header_row):
                cell = self._set_value(row=row_curr,
                                       col=len(left_dims) + 1 + i if left_dims else 2 + i,
                                       value=data_header_cell['value'],
                                       font=header_font,
                                       alignment=header_align)
                cell.border = self._create_border(positions=data_header_position,
                                                  style=self.BORDER_STYLE, color=self.BORDER_COLOR)

            row_curr += 1

        # Алгоритм слияния. Начинаем с первой строки. Запоминаем диапазон между ячейками типа 2,
        # параллельно объединяя слева направо ячейки с типом 2 и 1.
        # Затем переходим к следующей строке в рамках диапазона и объединяем слева направо ячейки с типом 2 и 1.

        # Для всех итогов делается слияние по столбцам

        def merge_cells(row_idx, col_idx_start, col_idx_end):

            if row_idx >= len(data_header):
                return

            start_range = None
            for col_idx in range(col_idx_start, col_idx_end):
                cell: dict = data_header[row_idx][col_idx]

                if cell['type'] in (2, 3, 5):
                    if start_range is None:
                        start_range = col_idx

                    else:
                        start_column = len(left_dims) + 1 + start_range if left_dims else (2 + start_range)
                        end_column = len(left_dims) + col_idx if left_dims else (1 + col_idx)
                        self.sheet.merge_cells(start_row=self.row_curr + row_idx,
                                               start_column=start_column,
                                               end_row=self.row_curr + row_idx,
                                               end_column=end_column)

                        # дополнительная проверка, если размерность не раскрыта и занимает ширину 1
                        merge_cells(row_idx + 1, start_range, col_idx)

                        if cell['type'] in (2, 3):
                            start_range = col_idx
                        else:
                            start_range = None

            # если колонки в диапазоне закончились, нужно объединить последний найденный диапазон
            if start_range is not None:
                self.sheet.merge_cells(start_row=self.row_curr + row_idx,
                                       start_column=len(left_dims) + 1 + start_range if left_dims else 2 + start_range,
                                       end_row=self.row_curr + row_idx,
                                       end_column=len(left_dims) + col_idx_end if left_dims else 1 + col_idx_end)
                merge_cells(row_idx + 1, start_range, col_idx_end)

        merge_cells(0, 0, len(data_header[0]))

        self.row_curr += len(data_header)

    @staticmethod
    def calculate_percent(row_idx: int,
                          cell_idx: int,
                          total_rows: int,
                          total_cells: int,
                          min_percent: int,
                          max_percent: int) -> int:
        total_steps = total_rows * total_cells
        current_step = row_idx * total_cells + cell_idx
        base_progress = current_step / total_steps
        scaled_progress = min_percent + base_progress * (max_percent - min_percent)
        return int(scaled_progress)

    @timing
    def _set_data(self, left_data: List[List[Dict]], data: List[List], left_dims: List, data_header: List[List[Dict]]):

        # set left data
        row_curr = self.row_curr
        left_data_alignment = self._create_alignment(align_h=self.LEFT_DIM_ALIGN_H,
                                                     align_v=self.LEFT_DIM_ALIGN_V,
                                                     wrap_text=self.LEFT_WRAP_TEXT)
        left_data_position = ('top', 'left', 'right', 'bottom')
        left_data_font = self._create_font()
        left_data_border = self._create_border(positions=left_data_position,
                                               style=self.BORDER_STYLE,
                                               color=self.BORDER_COLOR)

        for left_data_row in tqdm(left_data,
                                  desc="Putting left data rows to excel sheet",
                                  postfix=f"total progress: {LEFT_ROWS_PERCENT}%",
                                  file=sys.stdout):
            for i, left_data_cell in enumerate(left_data_row):
                cell = self._set_value(row_curr, i + 1, left_data_cell['value'],
                                       alignment=left_data_alignment, font=left_data_font)
                cell.border = left_data_border

            row_curr += 1

        self._merge_cells(left_data, left_dims)

        # set data
        row_curr = self.row_curr
        data_alignment = self._create_alignment(align_h=self.DATA_DIM_ALIGN_H,
                                                align_v=self.DATA_DIM_ALIGN_V,
                                                wrap_text=self.LEFT_WRAP_TEXT)
        data_position = ('top', 'left', 'right', 'bottom')
        data_font_row_total = self._create_font(font_bold=self.DATA_TOTAL_ROW_FONT_BOLD)
        data_font_col_total = self._create_font(font_bold=self.DATA_TOTAL_COL_FONT_BOLD)
        data_border = self._create_border(positions=data_position, style=self.BORDER_STYLE, color=self.BORDER_COLOR)

        measure_unit_map = {"thousand": 1000, "million": 1000000, "billion": 1000000000}
        previous_percent = DATA_ROWS_PERCENT_MIN
        for row_idx, data_row in enumerate(data):
            for cell_idx, data_cell in enumerate(data_row):
                data_number_format = self.DATA_NUMBER_FORMAT
                d_h = data_header[len(data_header) - 1][cell_idx]
                text_color = None
                if 'fact_format' in d_h:
                    fact_format = d_h['fact_format']

                    # применяем единицы измерения факта
                    if 'measureUnit' in fact_format:
                        measure_unit = measure_unit_map.get(fact_format.get('measureUnit', ""), "")
                        if data_cell is not None and measure_unit:
                            data_cell /= measure_unit

                    # формируем базовый формат данных, учитывая точность факта
                    if not data_number_format:
                        if 'precision' in fact_format:
                            precision = fact_format['precision']
                            data_number_format = '0.' + ('0' * precision) if precision > 0 else '0'
                        else:
                            data_number_format = '0.00'

                    # применяем разделитель разрядов
                    if 'split' in fact_format and fact_format.get('split', True) or 'split' not in fact_format:
                        data_number_format = '#,##' + data_number_format

                    # добавляем префикс и суффикс к базовому формату данных
                    base_format = data_number_format
                    prefix = fact_format.get('prefix') if 'prefix' in fact_format else None
                    suffix = fact_format.get('suffix') if 'suffix' in fact_format else None

                    # метод экранирования кавычек
                    def _esc_text(text):
                        return text.replace('"', '""')

                    if isinstance(prefix, str) and prefix.strip():
                        base_format = f'"{_esc_text(prefix)}"' + base_format
                    if isinstance(suffix, str) and suffix.strip():
                        base_format = base_format + f'"{_esc_text(suffix)}"'

                    data_number_format = base_format

                    # преобразуем формат цвета '#RRGGBB' -> 'FFRRGGBB'
                    color_hex = fact_format.get('color') if 'color' in fact_format else None
                    if isinstance(color_hex, str):
                        text_color = f"FF{color_hex.lstrip('#').upper()}"
                # select font; bold from col or row with totals
                if row_curr in self.total_row_sheet_indexes:
                    font = data_font_row_total
                elif len(left_dims) + 1 + cell_idx in self.total_col_sheet_indexes:
                    font = data_font_col_total
                else:
                    font = None

                # применяем цвет, если он задан
                if text_color is not None:
                    base_font = font or self._create_font()
                    font = Font(name=base_font.name, size=base_font.size, bold=base_font.bold, color=text_color)

                cell = self._set_value(row=row_curr,
                                       col=len(left_dims) + 1 + cell_idx if left_dims else 2 + cell_idx,
                                       value=data_cell,
                                       alignment=data_alignment,
                                       font=font)
                cell.border = data_border
                # cell.number_format = self.DATA_NUMBER_FORMAT
                cell.number_format = data_number_format

                percent = self.calculate_percent(row_idx,
                                                 cell_idx,
                                                 len(data),
                                                 len(data_row),
                                                 DATA_ROWS_PERCENT_MIN,
                                                 DATA_ROWS_PERCENT_MAX)
                if percent != previous_percent:
                    tqdm.write(f"Putting data rows to excel sheet, [total progress: {percent}%]")
                    previous_percent = percent

            row_curr += 1

        self.row_curr += len(left_data)

    @timing
    def put(self,
            top_dims: List,
            left_dims: List,
            data_header: List[List[Dict]],
            left_data: List[List[Dict]],
            data: List[List],
            view_format: Optional[CellAutoResizeFormat] = None,
            **kwargs):

        self.update_font_settings(**kwargs)

        self._set_headers(top_dims=top_dims, left_dims=left_dims, data_header=data_header)

        self._set_data(left_data=left_data, data=data, left_dims=left_dims, data_header=data_header)

        if view_format:
            view_format: dict
            view_format['row_end_num'] = self.row_curr
            CellAutoResizeBlock(self.sheet, self.row_start).put(**view_format)

        return self.row_curr + 1

    def _merge_cells(self, left_data, left_dims):
        """
        Алгоритм объединения ячеек в Excel:
        1. Объединяем пустые ячейки (value: None, type: 1) с предыдущим непустым значением
        2. Объединяем диапазон ячеек при встрече нового значения, отличного от текущего
        3. Для type == 5 (итоги) объединяем по горизонтали
        """

        def merge_cells_rec(col_idx, row_idx_start, row_idx_end):
            if col_idx >= len(left_dims):
                return

            # Сначала обрабатываем строки итогов (type == 5)
            for row_idx in range(row_idx_start, row_idx_end):
                cell = left_data[row_idx][col_idx]
                if cell['type'] == 5:
                    # Объединяем строку итогов по горизонтали
                    self._merge_range(
                        start_row=self.row_curr + row_idx,
                        start_column=col_idx + 1,
                        end_row=self.row_curr + row_idx,
                        end_column=len(left_dims)
                    )
                    self.total_row_sheet_indexes.add(self.row_curr + row_idx)

            # Теперь находим диапазоны объединения для текущего столбца
            start_range = None
            last_value = None
            last_type = None

            for row_idx in (
                    tqdm(range(row_idx_start, row_idx_end),
                         desc="Merging left data rows and columns in excel sheet",
                         postfix=f"total progress: {MERGED_LEFT_ROWS_PERCENT}%",
                         file=sys.stdout)
                    if col_idx == 0 else range(row_idx_start, row_idx_end)):
                cell = left_data[row_idx][col_idx]

                # Пропускаем строки с итогами, так как они уже обработаны
                if cell['type'] == 5:
                    # Завершаем текущий диапазон, если он есть
                    if start_range is not None and start_range < row_idx - 1:
                        self._merge_range(
                            start_row=self.row_curr + start_range,
                            start_column=col_idx + 1,
                            end_row=self.row_curr + row_idx - 1,
                            end_column=col_idx + 1
                        )
                        # Рекурсивно обрабатываем подуровни только если значение не пустое
                        if last_value is not None:
                            merge_cells_rec(col_idx + 1, start_range, row_idx)
                    start_range = None
                    last_value = None
                    last_type = None
                    continue

                # Пропускаем сгруппированные ячейки (type == 3) для рекурсивного вызова,
                # но включаем их в текущий диапазон объединения
                if cell['type'] == 3:
                    # Если это первая ячейка в столбце или предыдущая ячейка была сгруппированной
                    if start_range is None or last_type == 3:
                        start_range = row_idx
                        last_value = cell['value']
                        last_type = cell['type']
                        continue

                    # Завершаем текущий диапазон перед новой сгруппированной ячейкой
                    if start_range < row_idx - 1:
                        self._merge_range(
                            start_row=self.row_curr + start_range,
                            start_column=col_idx + 1,
                            end_row=self.row_curr + row_idx - 1,
                            end_column=col_idx + 1
                        )
                        # Рекурсивно обрабатываем подуровни
                        merge_cells_rec(col_idx + 1, start_range, row_idx)

                    # Начинаем новый диапазон с текущей ячейкой
                    start_range = row_idx
                    last_value = cell['value']
                    last_type = cell['type']
                    continue

                # Обработка пустых ячеек (value: None, type: 1)
                if cell['value'] is None and cell['type'] == 1:
                    # Если у нас уже есть начатый диапазон, продолжаем его
                    if start_range is not None:
                        continue

                    # Если это первая строка или нет предыдущего диапазона
                    # ищем предыдущую непустую ячейку
                    if row_idx > row_idx_start:
                        for prev_idx in range(row_idx - 1, row_idx_start - 1, -1):
                            prev_cell = left_data[prev_idx][col_idx]
                            if prev_cell['value'] is not None:
                                start_range = prev_idx
                                last_value = prev_cell['value']
                                last_type = prev_cell['type']
                                break
                else:
                    # Если это новое значение, отличное от текущего
                    if start_range is not None and (cell['value'] != last_value or cell['type'] != last_type):
                        # Завершаем предыдущий диапазон
                        if start_range < row_idx - 1:
                            self._merge_range(
                                start_row=self.row_curr + start_range,
                                start_column=col_idx + 1,
                                end_row=self.row_curr + row_idx - 1,
                                end_column=col_idx + 1
                            )
                            # Рекурсивно обрабатываем подуровни
                            merge_cells_rec(col_idx + 1, start_range, row_idx)

                        # Начинаем новый диапазон
                        start_range = row_idx
                        last_value = cell['value']
                        last_type = cell['type']
                    elif start_range is None:
                        # Начинаем новый диапазон
                        start_range = row_idx
                        last_value = cell['value']
                        last_type = cell['type']

            # Завершаем последний диапазон в столбце
            if start_range is not None and start_range < row_idx_end - 1:
                self._merge_range(
                    start_row=self.row_curr + start_range,
                    start_column=col_idx + 1,
                    end_row=self.row_curr + row_idx_end - 1,
                    end_column=col_idx + 1
                )
                # Рекурсивно обрабатываем подуровни для последнего диапазона
                if last_value is not None:
                    merge_cells_rec(col_idx + 1, start_range, row_idx_end)

        # Запускаем рекурсивную обработку с первого столбца
        merge_cells_rec(0, 0, len(left_data))

    def _merge_range(self, start_row, start_column, end_row, end_column):
        """
        Если MERGE_LEFT_DIMS_CELLS true, то мержим клетки в одну большую клетку,
        иначе заполняем клетку одним и тем же значением из первой клетки
        """
        if start_row > end_row or start_column > end_column:
            LOGGER.info(f"Invalid merge range: {start_row}:{start_column} to {end_row}:{end_column}")
            return

        def check_ranges_overlap(range1, range2):
            r1_start_row, r1_start_col, r1_end_row, r1_end_col = range1
            r2_start_row, r2_start_col, r2_end_row, r2_end_col = range2

            return not (r1_end_row < r2_start_row or
                        r1_start_row > r2_end_row or
                        r1_end_col < r2_start_col or
                        r1_start_col > r2_end_col)

            # Проверка пересечений с существующими объединениями
        current_range = (start_row, start_column, end_row, end_column)
        for merged_range in self.sheet.merged_cells.ranges:
            existing_range = (merged_range.min_row, merged_range.min_col,
                              merged_range.max_row, merged_range.max_col)
            if check_ranges_overlap(current_range, existing_range):
                LOGGER.info(f"Merge range {current_range} overlaps with existing: {existing_range}")
                return

        if TableConf.MERGE_LEFT_DIMS_CELLS:
            self.sheet.merge_cells(start_row=start_row, start_column=start_column,
                                   end_row=end_row, end_column=end_column)

        else:
            value = self.sheet.cell(start_row, start_column).value

            for y in range(start_row, end_row + 1):
                for x in range(start_column, end_column + 1):
                    self.sheet.cell(y, x, value)
