import subprocess
import argparse
import os

# временный парсинг только пути до конфига
parser = argparse.ArgumentParser()
parser.add_argument('--config_path')
args, _ = parser.parse_known_args()

if args.config_path:
    os.environ["CONFIG_PATH"] = args.config_path

from typing import Optional

from poly_connector.connector import Connector
from poly_connector.load_data import DataLoader

from excel.editor import ExcelEditor
from excel.blocks import *

from excel.data_processing.hiararchy import Hierarchy

from tools.utils import merge_dicts, crop_extra_double_quotes, timing

# bl
from polymatica.business_scenarios import BusinessLogic
# connecton
from settings import URL, LOGIN_AUTH, SESSION_AUTH, SCRIPT_NAME, MODULE_UUID
# paths
from settings import OUTPUT_FILENAME, ROOT_PATH
# columns
from settings import COL_AUTOSIZE, COL_MIN_WIDTH, COL_MAX_WIDTH
# hierarchy_columns
from settings import HIERARCHY_COLUMNS, SHIFT_SYMB
# block
from settings import DataConf, DataRelevanceConf, TitleConf, TableConf
# logger
from settings import LOGGER

version = "1.4.0"


class Exporter:
    """
    Export data from polymatica analytics.
    """

    def __init__(self, connector: Connector,
                 script_name: Optional[str] = SCRIPT_NAME,
                 module_uuid: Optional[str] = MODULE_UUID):
        self.module_uuid = module_uuid
        self.script_name = script_name
        self._connector = connector
        self.conn: BusinessLogic = self._connector.conn
        self.data_loader = DataLoader(conn=self.conn,
                                      script_name=self.script_name,
                                      module_uuid=self.module_uuid)
        self._validate()

    def _validate(self):
        pass

    @timing
    def export2excel(self, args: argparse.Namespace, show_full: bool = True):
        """
        Export data to xlsx format.

        :param args: argparser
        :param show_full: show all table headers
        """
        top_dims, left_dims, data_header, left_data, data, extra_info = self.data_loader.read_data_full()
        data_cols_count, data_rows_count, inter_total_hidden_dimensions, show_global_horz_total, \
            show_global_vert_total = (extra_info[key] for key in (
            "data_cols_count", "data_rows_count", "inter_total_hidden_dimensions", "show_global_horz_total",
            "show_global_vert_total"))
        top_dims_count, left_dims_count = self.data_loader.top_dims, self.data_loader.left_dims,

        LOGGER.info(f"Left dims count: {left_dims_count}; top dims count: {top_dims_count}; header columns count: {data_cols_count}; rows count: {data_rows_count}")

        # иерархия
        if HIERARCHY_COLUMNS:
            if top_dims_count != 0:
                raise AssertionError("There should be no top dimensions in the multisphere")
            if inter_total_hidden_dimensions:
                raise AssertionError("There should be no hidden subtotals in the multisphere")
            if show_global_horz_total is False:
                raise AssertionError("There should be horizontal totals in the multisphere")
            if show_global_vert_total is False:
                raise AssertionError("There should be vertical totals in the multisphere")
            top_dims, left_dims, data_header, left_data, data = Hierarchy(
                grouped_column_ids=args.hierarchy_columns,
                tab_symb=args.shift_symb
            ).process(top_dims, left_dims, data_header, left_data, data)
            data_rows_count = len(data)
            left_dims_count = len(left_dims)

        dims_filter_splited = self.data_loader.get_dimension_with_filters_split_by_position()
        cube_info = self.data_loader.get_cube_info()
        cube_name = cube_info['name']
        title = TitleConf.NAME or (self.conn.get_active_layer_name() if not TitleConf.OLAP_NAME_INSTEAD_OF_LAYER_NAME
                                   else cube_name)

        if self.script_name:
            modules = self.conn.get_module_list()
            module_uuid, _, _, _ = modules[0]
            self.module_uuid = module_uuid
        module_setting = self.conn.execute_manager_command(
            command_name="user_iface", state="load_settings", module_id=self.module_uuid)
        module_info = self.conn.h.parse_result(result=module_setting, key="settings") or dict()
        facts_format = module_info.get('config_storage', {}).get('facts-format', {}).get('__suffixes', {})
        for d_h in data_header:
            for d_h2 in d_h:
                if d_h2.get('fact_id') is not None:
                    fact_format = facts_format.get(d_h2['fact_id'], [])
                    d_h2['fact_format'] = fact_format
                    if 'measureUnit' in fact_format:
                        if fact_format['measureUnit'] == 'thousand':
                            d_h2['value'] += ', тыс.'
                        if fact_format['measureUnit'] == 'million':
                            d_h2['value'] += ', млн'
                        if fact_format['measureUnit'] == 'billion':
                            d_h2['value'] += ', млрд'

        excel_editor = ExcelEditor().create()

        blocks = [DateBlock]
        if cube_info["relevance_date"] != "":
            blocks.append(DateRelevanceBlock)
        blocks.append(TitleBlock)
        blocks.append(OutFilterBlock)
        blocks.append(DataFilterBlock)
        blocks.append(TableBlock)
        
        formats = [DateFormat(date_format=args.date_format)]
        if cube_info["relevance_date"] != "":
            formats.append(DateRelevanceFormat(relevance_date=cube_info["relevance_date"]))
        formats.append(TitleFormat(name=title,
                        columns_size=(len(data_header) + len(left_dims) - 1)))
        formats.append(OutFilterFormat(filters=dims_filter_splited.get('out'),
                            columns_size=(len(data_header) + len(left_dims) - 1)))
        formats.append(DataFilterFormat(filters=merge_dicts(dims_filter_splited.get('left'), dims_filter_splited.get('up')),
                             columns_size=(len(data_header) + len(left_dims) - 1)))
        formats.append(TableFormat(top_dims=top_dims,
                                    left_dims=left_dims,
                                    data_header=data_header,
                                    left_data=left_data,
                                    data=data,
                                    view_format=CellAutoResizeFormat(
                                        autosize=args.col_autosize,
                                        min_width=args.col_min_width,
                                        max_width=args.col_max_width)))

        excel_editor.add_blocks(blocks=blocks, formats=formats)

        # save file
        path_filename_output = crop_extra_double_quotes(args.path_output)

        filename_output = os.path.basename(path_filename_output)
        path_output = os.path.dirname(path_filename_output)
        filename_output_without_ext, filename_output_ext = filename_output.split('.')

        if path_output:
            os.makedirs(path_output, exist_ok=True)

        if filename_output_ext == 'xlsx':
            excel_editor.save(path_filename_output)
        elif filename_output_ext == 'ods':
            tmp_path_filename_output = os.path.join(path_output, filename_output_without_ext + '.xlsx')
            excel_editor.save(tmp_path_filename_output)
            subprocess.run(["libreoffice",
                            "--headless",
                            "--invisible",
                            "--convert-to", "ods",
                            f"--outdir", path_output,
                            tmp_path_filename_output])
            os.remove(tmp_path_filename_output)
        else:
            raise ValueError(f"Not valid filename extension: {filename_output_ext}; available 'xlsx' or 'ods'")

        if not os.path.isfile(path_filename_output):
            raise OSError(f"Couldn't save file: {path_filename_output}")

        LOGGER.info(f"Saved file: {path_filename_output}")

    def __del__(self):
        del self.data_loader
        # self.conn.close_current_cube()  # not use when not run scenario!
        del self.conn


def create_parser() -> argparse.ArgumentParser:
    """
    Create argparser.
    """
    parser = argparse.ArgumentParser(
        prog='main.py',
        description='Python script for exporting tables from analytics to excel. '
                    'All arguments have default value from config.json!')

    # connection
    parser.add_argument("--connection_url", type=str, action='store', help='Analytics url',
                        default=URL)
    parser.add_argument("--connection_session", type=str, action='store', nargs='+',
                        help="Analytics session args: 'session_id', 'manager_uuid' and 'full_polymatica_version'",
                        default=list(SESSION_AUTH.values()))
    parser.add_argument("--connection_login", type=str, action='store', nargs='+',
                        help="Analytics login args: 'username' and 'password'",
                        default=list(LOGIN_AUTH.values()))
    parser.add_argument("--connection_script_name", type=str, action='store', help='Analytics script name',
                        default=SCRIPT_NAME)
    parser.add_argument("--olap_module", type=str, action='store', help='Analytics module uuid',
                        default=MODULE_UUID)
    # paths
    parser.add_argument("--path_output", type=str, action='store',
                        help='Output path with filename with extension, example: "result.xlsx", "result.ods"',
                        default=OUTPUT_FILENAME)
    parser.add_argument('--config_path', default=os.path.join(ROOT_PATH, 'config.json'))


    # columns
    parser.add_argument("--col_autosize", type=bool, action='store', help='Columns auto resize, example: True',
                        default=COL_AUTOSIZE)
    parser.add_argument("--col_min_width", type=int, action='store', help='Columns min width, example: 5',
                        default=COL_MIN_WIDTH)
    parser.add_argument("--col_max_width", type=int, action='store', help='Columns max width, example: 30',
                        default=COL_MAX_WIDTH)

    # data hierarchy
    parser.add_argument("--hierarchy_columns", type=list, action='store',
                        help='Hierarchy columns; example: "[[1, 3, 5]]"',
                        default=HIERARCHY_COLUMNS or [])
    parser.add_argument("--shift_symb", type=str, action='store',
                        help='Shift symbol or string; example: "\\t"',
                        default=SHIFT_SYMB)

    # date format
    parser.add_argument("--date_format", type=str, action='store',
                        help='Datetime format; example: "%%d.%%m.%%Y"',
                        default=DataConf.DATE_FORMAT)
    
    # date relevance format
    # parser.add_argument("--date_relevance_format", type=str, action='store',
    #                     help='Datetime format; example: "%%d.%%m.%%Y"',
    #                     default=DataRelevanceConf.DATE_FORMAT)

    # table
    parser.add_argument("--table_merge_left_dims", type=bool, action='store',
                        help='Merge left dims in table; example: True',
                        default=TableConf.MERGE_LEFT_DIMS_CELLS)

    return parser


def main():
    parser = create_parser()
    args = parser.parse_args()
    LOGGER.info("Login connection" if args.connection_login else "Session connection")
    CONN_LOGIN = {key: crop_extra_double_quotes(val) for key, val in
                  zip(('username', 'password'), args.connection_login)}
    CONN_SESS = {key: crop_extra_double_quotes(arg) for key, arg in
                 zip(("session_id", "manager_uuid", "full_polymatica_version"), args.connection_session)} if isinstance(
        args.connection_session, (list, tuple)) else args.connection_session
    connector = Connector(host=args.connection_url,
                          login=CONN_LOGIN.get('username', None), password=CONN_LOGIN.get('password', None),
                          session_auth=CONN_SESS)
    connector.connect()
    LOGGER.info("Success connect")
    assert args.connection_script_name is None or args.olap_module is None, \
        "Set only script name or module uuid, don't use both params"
    exporter = Exporter(connector=connector,
                        script_name=crop_extra_double_quotes(args.connection_script_name),
                        module_uuid=crop_extra_double_quotes(args.olap_module))
    exporter.export2excel(args)


if __name__ == "__main__":
    main()
