import sys
from polymatica.business_scenarios import GetDataChunk, BusinessLogic
from itertools import compress
from tqdm import tqdm
from typing import Optional, Tuple, List, Dict

from settings import SCRIPT_NAME, MODULE_UUID, TableConf, UNITS
from tools.utils import timing

# logger
from settings import LOGGER

LOADING_DATA_PERCENT = 5

class DataLoader:
    """
    Load data from analytics.
    """

    DIM_POSITIONS = ('out', 'left', 'up')
    DIM_DISABLE_POS = (DIM_POSITIONS[0],)
    DIM_ACTIVE_POS = (DIM_POSITIONS[1], DIM_POSITIONS[2])

    def __init__(self, conn: Optional[BusinessLogic],
                 script_name: Optional[str] = SCRIPT_NAME,
                 module_uuid: Optional[str] = MODULE_UUID):
        self.conn: Optional[BusinessLogic] = conn
        self.data_connect: Optional[GetDataChunk] = None

        self.left_dims = None
        self.top_dims = None
        self.fact_dims = None

        self._olap_module_info: Optional[dict] = None

        if script_name:
            self.prepare_data(script_name=script_name)
        elif module_uuid:
            self.prepare_data(module_uuid=module_uuid)

    def _get_script_by_name(self, script_name: str) -> dict:
        """
        Find script by name.

        :param script_name: script name (scenario name)
        :return: dict with script params
        """
        scripts: list = self.conn.get_scripts_list()
        if script_name:
            scripts = [script for script in self.conn.get_scripts_list() if script['name'] == script_name]
        if not scripts:
            raise ValueError(f"Can't find script with name {script_name}")
        script = scripts[0]
        return script

    def prepare_data(self,
                     script_name: Optional[str] = None,
                     module_uuid: Optional[str] = None):
        """
        Prepare data using one way:
            - search script by name, load it, get column names and number of rows;
            - load data by id opened module

        :param script_name: script name (scenario name)
        :param module_uuid: module uuid
        """
        if script_name:
            script = self._get_script_by_name(script_name)
            self.conn.run_scenario(script['id'])
        elif module_uuid:
            self.conn.set_multisphere_module_id(module_uuid)
            # res = self.conn.execute_olap_command(command_name="dimension", state="list_rq",)
        else:
            raise ValueError("Set script name or module id")

        self.data_connect = GetDataChunk(self.conn)

        self.left_dims = self.data_connect.left_dims_qty
        self.top_dims = self.data_connect.top_dims_qty
        self.fact_dims = self.data_connect.facts_qty

    @timing
    def read_data_full(self, replace_total=True) -> Tuple[List, List, List[List], List[List], List[List], Dict]:
        """
        Read all rows from sphere.

        :return: top dims list, left dims list, data header list[list], left data list[list], data list[list],
            extra info dict
        """

        def replace_value_for_types(data: List[List[Dict]], src_type: int, dst_value: str):
            for row in data:
                for elem in row:
                    if elem['type'] == src_type:
                        elem['value'] = dst_value
            return data

        top_dims, left_dims, data_header, load_data_by_chunk, extra_info = \
            self.data_connect.load_sphere_chunk_v2(units=UNITS)
        data_header = replace_value_for_types(data_header, 5, TableConf.TOTAL_NAME)
        data_rows_count = extra_info['data_rows_count']

        left_data, data = [], []
        for _left_data, _data in tqdm(
                load_data_by_chunk(),
                total=(data_rows_count // UNITS + (1 if data_rows_count % UNITS else 0)),
                desc="Loading data from analytica", postfix=f"total progress: {LOADING_DATA_PERCENT}%",
                file=sys.stdout):
            if replace_total:
                _left_data = replace_value_for_types(_left_data, 5, TableConf.TOTAL_NAME)
            left_data += _left_data
            data += _data

        return top_dims, left_dims, data_header, left_data, data, extra_info

    def _get_dimension_with_filters(self) -> list:
        """
        Get dimensions with filters.

        :return: list with dicts
        """
        return self.data_connect.get_dimension_with_filters()

    def get_dimension_with_filters_split_by_position(self) -> dict:
        """
        Get dimensions with filters and split them by key 'position'.

        :return: dict with all unique 'position' values and dims
        """

        dims_splited = {pos: {} for pos in self.DIM_POSITIONS}

        for dim in self._get_dimension_with_filters():
            name = dim['name']
            split_p = dim['position']

            filter_values = dim['filter']['values']
            filter_mask = dim['filter']['mask']

            dims_splited[split_p][name] = tuple(compress(filter_values, filter_mask))

        response = self.data_connect.sc.execute_olap_command(
            command_name="dimension",
            state="list_rq",
        )
        command_data: dict = response['queries'][0]['command']
        filter_mode: bool = command_data['filter_mode']
        if not filter_mode:
            dims_splited['out'] = {}

        return dims_splited

    def get_cube_info(self) -> dict:
        """
        Get info about current sphere.

        :return:
        """
        try:
            return self.conn.get_cube_info()
        except Exception as e:
            LOGGER.info("Can't get cube info, use name in config")
            LOGGER.info(f"Exception: {e}")
            return {}

    @property
    def visible_measures(self):
        return self.data_connect.get_measure_dims(visible=True)


def recursive_flatten(arr):
    new_arr = []
    for idx, val in enumerate(arr):
        if isinstance(val, (tuple, list, set)):
            tmp_arr = recursive_flatten(val)
            new_arr += tmp_arr
        else:
            new_arr += [val]
    return new_arr
