from openpyxl.styles import Font, Alignment, Border, Side
from typing import Optional, Sequence

from settings import DEFAULT_FONT_SIZE, DEFAULT_FONT_NAME


class BaseBlock:

    BORDER_POSSIBLE_POSITIONS = {'bottom', 'left', 'right', 'top'}

    def __init__(self, sheet, row_start: int = 1):
        self.sheet = sheet
        self.row_start = row_start
        self.row_curr = self.row_start

    @staticmethod
    def _create_font(font_size: Optional[int] = DEFAULT_FONT_SIZE,
                     font_bold: Optional[bool] = None,
                     font_name: Optional[str] = DEFAULT_FONT_NAME) -> Font:
        """
        Create Font object for openpyxl cell.

        :param font_size: optional font size
        :param font_bold: optional font bold
        :param font_name: optional font name, example: calibri, arial
        :return: Font object
        """
        return Font(size=font_size, bold=font_bold, name=font_name)

    @staticmethod
    def _create_alignment(align_h: Optional[str] = None, align_v: Optional[str] = None,
                          wrap_text: Optional[bool] = None,) -> Alignment:
        """
        Create Alignment object for openpyxl cell.

        :param align_h: optional horizontal alignment
        :param align_v: optional vertical alignment
        :param wrap_text: optional wrap test on next line into cell
        :return: Alignment object
        """
        return Alignment(horizontal=align_h, vertical=align_v, wrap_text=wrap_text)

    def _get_value_cell(self, row: int, col: int):
        """
        Get cell value by cell position.

        :param row: row num
        :param col: col num
        :return: cell value
        """
        return self.sheet.cell(row=row, column=col).value

    def _set_value(self, row: int, col: int, value: str,
                   font: Optional[Font] = None, alignment: Optional[Alignment] = None):
        """
        Set value in excel-sheet with default font and alignment params.

        :param row: row num
        :param col: col num
        :param value: string text
        :param font: optional font style
        :param alignment: optional alignment value in cell
        :returns: modified cell
        """
        cell = self.sheet.cell(row=row, column=col, value=value)
        cell.font = font or self._create_font()
        cell.alignment = alignment or self._create_alignment()
        return cell

    @classmethod
    def _create_border(cls, positions: Sequence[str], style: Optional[str] = None,
                       color: Optional[str] = None) -> Border:
        """
        Create Border object for openpyxl cell.

        :param positions: sequence with positions
        :param style: optional border style
        :param color: optional string format color, read more:
         https://openpyxl.readthedocs.io/en/latest/styles.html#indexed-colours
        :return: Border object
        """
        side = Side(style=style, color=color)
        param_dict = {}
        for position in positions:
            if position not in cls.BORDER_POSSIBLE_POSITIONS:
                raise ValueError(f"Not correct position, available {cls.BORDER_POSSIBLE_POSITIONS}")
            param_dict[position] = side
        return Border(**param_dict)

    @classmethod
    def update_font_settings(cls, *args, **kwargs) -> None:
        """
        Update font settings for current block.
        """
        pass

    def put(self, *args, **kwargs) -> int:
        """
        Put block in xlsx sheet.

        :returns: new row index
        """
        raise NotImplementedError
